/* _kblinux.h -- Linux keyboard handler installation
 * Copyright (C) Markus F.X.J. Oberhumer, Harm Hanemaayer and others
 * For conditions of distribution and use, see copyright notice in kb.h 
 */

/* WARNING: this file should *not* be used by applications. It is
   part of the implementation of the keyboard library and is
   subject to change. Applications should only use kb.h.
 */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <linux/kd.h>
#include <linux/vt.h>


/* IMPORTANT NOTE: you have to update the keyboard manually under Linux 
 *                 !!! there are NO INTERRUPTS !!! 
 *
 * see also: svgalib 1.2.9: src/keyboard/keyboard.c
 * see also: kbd 0.91: src/showkey.c
 */


/***********************************************************************
// 
************************************************************************/

static int get_fd_info(int fd, int *mode, struct termios *t)
{
	int dummy_mode;
	struct termios dummy_termios;

	if (fd < 0)
		return -1;

	if (mode == NULL)
		mode = &dummy_mode;
	if (t == NULL)
		t = &dummy_termios;

	if (ioctl(fd, KDGKBMODE, mode) != 0)
	{
#if defined(KB_DEBUG)
		perror("libkb: cannot get keyboard mode");
#endif
		return -1;
	}
#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
	fprintf(stderr,"libkb info: fd: %d ioctl: kbmode=%d\n", fd, *mode);
#endif

	if (tcgetattr(fd, t) != 0)
	{
#if defined(KB_DEBUG)
		perror("libkb: tcgetattr");
#endif
		return -1;
	}
#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
	fprintf(stderr,"libkb info: fd: %d tcgetattr: iflag=0x%04lx "
		"oflag=0x%04lx lflag=0x%04lx %d %d\n", 
		fd, t->c_iflag, t->c_oflag, t->c_lflag, t->c_cc[VMIN], t->c_cc[VTIME]);
#endif

	return 0;
}


/***********************************************************************
// 
************************************************************************/

static int _my_kbd_fd = -1;
static int _my_vt = -1;

static int oldkbmode = K_XLATE;
static struct termios oldkbdtermios;

static int keyboard_init_return_fd(void)
{
	int r = 0;
	struct termios newkbdtermios;
	struct vt_stat vtinfo;
	long first_non_opened;


/* STEP 1a: open /dev/console */
	if (_my_kbd_fd == -1)
		_my_kbd_fd = open("/dev/console", O_RDONLY);
	if (_my_kbd_fd == -1)
	{
#if defined(KB_DEBUG)
		perror("libkb: cannot open /dev/console");
#endif
		return -1;
	}
	if (_my_kbd_fd < 0)
	{
#if defined(KB_DEBUG)
		fprintf(stderr,"libkb: fd: %d, strange\n",_my_kbd_fd);
#endif
		return -1;
	}


/* STEP 1b: get current settings */
	if (get_fd_info(_my_kbd_fd, &oldkbmode, &oldkbdtermios) != 0)
		return -1;
#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
	if (oldkbmode != K_XLATE)	/* what about K_UNICODE ? */
		fprintf(stderr,"libkb info: keyboard not in K_XLATE, strange\n");
#endif


/* STEP 1c: get my VT info */
	if (ioctl(_my_kbd_fd, VT_GETSTATE, &vtinfo) != 0)
	{
#if defined(KB_DEBUG)
		perror("libkb: ioctl VT_GETSTATE");
#endif
		return -1;
	}
#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
	fprintf(stderr,"libkb info: fd: %d vt_stat: v_active=%d "
		"v_signal=0x%x v_state=0x%x\n", _my_kbd_fd, (int) vtinfo.v_active,
		(unsigned) vtinfo.v_signal, (unsigned) vtinfo.v_state);
#endif
	_my_vt = vtinfo.v_active;


/* STEP 1d: get number of virtual terminals */
	if (ioctl(_my_kbd_fd, VT_OPENQRY, &first_non_opened) != 0)
	{
#if defined(KB_DEBUG)
		perror("libkb: ioctl VT_OPENQRY");
#endif
		return -1;
	}
#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
	fprintf(stderr,"libkb info: first non opened VT: %ld\n",
		first_non_opened);
#endif
	if (_my_vt <= 0 || _my_vt >= first_non_opened)
		return -1;



/* STEP 2a: set termios */
	newkbdtermios = oldkbdtermios;
#if 1
	newkbdtermios.c_iflag = 0;
#else
	newkbdtermios.c_iflag &= ~(ISTRIP | IGNCR | ICRNL | INLCR | IXOFF | IXON);
#endif
	newkbdtermios.c_lflag &= ~(ISIG | ICANON | ECHO);
	/* Making these 0 seems to have the desired effect. */
	newkbdtermios.c_cc[VMIN] = 0;
	newkbdtermios.c_cc[VTIME] = 0;

	if (tcsetattr(_my_kbd_fd, TCSANOW, &newkbdtermios) != 0)
	{
		r = -2;
#if defined(KB_DEBUG)
		perror("libkb: tcsetattr");
#endif
	}


/* STEP 2b: set kbmode */
	if (ioctl(_my_kbd_fd, KDSKBMODE, K_RAW) != 0)
	{
		r = -2;
#if defined(KB_DEBUG)
		perror("libkb: ioctl KDSKBMODE K_RAW");
#endif
	}


/* all done */
	if (r < 0)
		return r;
#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
	get_fd_info(_my_kbd_fd, NULL, NULL);	/* print the new settings */
#endif
	return _my_kbd_fd;	/* OK, return fd */
}


/***********************************************************************
// 
************************************************************************/

static void _kb_remove(int final)
{
	int r = 0;

	if (_my_kbd_fd < 0)
		return;

#if defined(KB_DEBUG)
	fprintf(stderr,"_kb_remove 1: fd: %d  final: %d\n", _my_kbd_fd, final);
	get_fd_info(_my_kbd_fd, NULL, NULL);	/* print the current settings */
#endif

	if (ioctl(_my_kbd_fd, KDSKBMODE, oldkbmode) != 0)
		r |= 1;
	if (tcsetattr(_my_kbd_fd, 0, &oldkbdtermios) != 0)
		r |= 2;
#if defined(KB_DEBUG) && (KB_DEBUG >= 2)
	get_fd_info(_my_kbd_fd, NULL, NULL);	/* print the current settings */
#endif

	if (final)
	{
		if (close(_my_kbd_fd) != 0)
			r |= 4;
		_my_kbd_fd = -1;
	}

#if defined(KB_DEBUG)
	fprintf(stderr,"_kb_remove 2: fd: %d  error code: %d\n", _my_kbd_fd, r);
#endif
}


static int _kb_install(void)
{
	int fd = keyboard_init_return_fd();
	if (fd == -1)
		return -1;
	else if (fd < 0)
	{
		_kb_remove(1);
		return -1;
	}
	else
		return 0;
}


/***********************************************************************
// virtual terminal switching
// termios should be saved because svgalib enables ISIG in c_lflag
************************************************************************/

static void kb_switch_vt(int vt)
{
	long first_non_opened;
	struct termios t;
	int r;

	if (vt <= 0 || vt == _my_vt)
		return;

	/* get number of virtual terminals */
	if (ioctl(_my_kbd_fd, VT_OPENQRY, &first_non_opened) != 0)
	{
#if defined(KB_DEBUG)
		perror("libkb VT: ioctl VT_OPENQRY");
#endif
		return;
	}
	if (vt >= first_non_opened)
		return;

	/* save current termios */
	r = tcgetattr(_my_kbd_fd, &t);
#if defined(KB_DEBUG)
	if (r != 0)
		perror("libkb VT: tcgetattr");
#endif
#if defined(KB_DEBUG) && (KB_DEBUG >= 3)
	if (r == 0)
		fprintf(stderr,"libkb VT info: fd: %d tcgetattr: "
			"iflag=0x%04lx oflag=0x%04lx lflag=0x%04lx %d %d\n", 
			_my_kbd_fd, t.c_iflag, t.c_oflag, t.c_lflag,
			t.c_cc[VMIN], t.c_cc[VTIME]);
#endif

#if 0
	if (_kb_flags & KB_FLAG_EMERGENCY_SIGALRM)
		_kb_signal_alarm_pause();
#endif

	/* Do the switching.
	 * This will also generate a signal catched by svgalib to restore textmode.
 	 */
	if (ioctl(_my_kbd_fd, VT_ACTIVATE, vt) != 0)
	{
#if defined(KB_DEBUG)
		perror("libkb VT: ioctl VT_ACTIVATE");
#endif
	}

	/* restore termios */
	if (r == 0)
	{
		r = tcsetattr(_my_kbd_fd, TCSANOW, &t);
#if defined(KB_DEBUG)
		if (r != 0)
			perror("libkb VT: tcsetattr");
#endif
	}
}


/***********************************************************************
// Only switch if Alt and not more than one Shift key is pressed.
//
// We are more restrictive than the standard behavior because 
// we try to avoid the situation that a now pressed key is released
// after the terminal switch and we still think it is pressed 
// when we switch back to here.
// This means we prefer missing a key-press to missing a key-release
// in a different console.
************************************************************************/

static int is_switch(int *vt, int vtnum)
{
	int shift;

	if (!_kb_key[KB_SCAN_ALT])
		return 0;

#if 1
	/* while libkb is still in beta, we always do the switch if SCRLOCK
	 * is pressed. If there is a bug and _kb_keys_pressed gets out
	 * of sync, no switching would be possible.
	 */
	if (_kb_key[KB_SCAN_SCRLOCK])
	{
		*vt = (_kb_key[KB_SCAN_LSHIFT] || _kb_key[KB_SCAN_RSHIFT])
				? vtnum + 12 : vtnum;
		return 1;
	}
#endif

	if (_kb_keys_pressed > 2)
		return 0;
	shift = 0;
	if (_kb_key[KB_SCAN_LSHIFT])  shift++;
	if (_kb_key[KB_SCAN_RSHIFT])  shift++;
	if (_kb_keys_pressed - shift != 1)
		return 0;
	
	/* a valid key combination */
	*vt = shift ? vtnum + 12 : vtnum;
	return 1;
}


/***********************************************************************
// update the keyboard
************************************************************************/

void kb_update(void)
{
	unsigned char buf[128];
	unsigned char *p;
	int bytesread;

	if (!_kb_mode)
		return;
	
	if (_kb_flags & KB_FLAG_LINUX_NO_VT)
	{
		/* optimized version for no virtual terminal switching */
		while ((bytesread = read(_my_kbd_fd, buf, sizeof(buf))) > 0)
			for (p = buf; p < &buf[bytesread]; p++)
				_my_raw_handler(*p);
	}
	else
	{
		/* enable virtual terminal switching */
		int vt = 0;

		while ((bytesread = read(_my_kbd_fd, buf, sizeof(buf))) > 0)
			for (p = buf; p < &buf[bytesread]; p++)
			{
				int scan = *p;

				if (scan >= KB_SCAN_F1 && scan <= KB_SCAN_F12)
				{
					if (scan <= KB_SCAN_F10)
					{
						if (is_switch(&vt, scan - (KB_SCAN_F1 - 1)))
							continue;		/* ignore this keypress */
					}
					else if (scan >= KB_SCAN_F11)
					{
						if (is_switch(&vt, scan - (KB_SCAN_F11 - 11)))
							continue;		/* ignore this keypress */
					}
					else if (scan == KB_SCAN_LAST_CONSOLE)
					{
						/* TODO: is there a system call ? */
					}
				}

				_my_raw_handler(scan);
			}

		if (vt > 0)
			kb_switch_vt(vt);
	}
	

	/* Control-C check */
	if (_kb_flags & KB_FLAG_SIGINT)
	{
		if (_kb_key[KB_SCAN_C] && 
			(_kb_key[KB_SCAN_LCONTROL] || _kb_key[KB_SCAN_RCONTROL]))
			raise(SIGINT);
	}

	if (_kb_flags & KB_FLAG_EMERGENCY_SIGALRM)
		_kb_signal_alarm_update();
}


/*
vi:ts=4
*/

